#include "HardwareProfile.h"
#include "mystring.h"
#include "clock.h"
#include "keys.h"
#include "led.h"
#include "main.h"

/* 
	
	Clock System, by Mauro Grassi, 2010.

*/

#pragma udata clockdata

volatile TIME_T 		systemTime;
volatile unsigned int 	timer1OverFlow;
volatile unsigned long  timeAbsolute;
volatile unsigned char  alarmEvent;
volatile unsigned long  alarmAbsolute;
volatile unsigned char	timeUp;

#pragma code usercode

const rom unsigned int monthDayLimits[12]=
{
31, 
28, 
31, 
30, 
31, 
30, 
31, 
31, 
30, 
31, 
30,
31 
};

const rom int monthDays[11]=
{
31, 
59, 
90, 
120, 
151, 
181, 
212, 
243, 
273, 
304, 
334, 
};

const rom char monthNames[12][4]
={ 	"Jan",
	"Feb",
	"Mar",
	"Apr",
	"May",
	"Jun",
	"Jul",
	"Aug",
	"Sep",
	"Oct",
	"Nov",
	"Dec"
};

const rom char weekDays[7][4]={ 	
	"Mon",
	"Tue",
	"Wed",
	"Thu",
	"Fri",
	"Sat",
	"Sun",
};

void addMonths(TIME_T* argTimeOutput, long n)
{
	/* adds and roughly clips the output */
	if(n>0)
	{
	n+=(long)argTimeOutput->month;
	if(n>0)n--;
	argTimeOutput->year+=(unsigned short)(n/12);
	argTimeOutput->month=(n % 12)+1;
	}
}

void addDays(TIME_T* argTimeOutput, long n)
{
	if(n>0)
	{
	n+=(long)argTimeOutput->day;
	if((n>=1)&&(n<(long)daysInMonth(argTimeOutput->month, argTimeOutput->year)))
	{

	}
	else
	if(n<1)
	{
		n=1;
	}
	else
	{
		n=daysInMonth(argTimeOutput->month, argTimeOutput->year);
	}
	argTimeOutput->day=(unsigned char)n;
	}
}

void addHours(TIME_T* argTimeOutput, long n)
{
	if(n>0)
	{
	n+=(long)argTimeOutput->hours;
	argTimeOutput->hours=(n % 24);
	addDays(argTimeOutput, (n/24));
	}
}

void addMinutes(TIME_T* argTimeOutput, long n)
{
	if(n>0)
	{
	n+=(long)argTimeOutput->mins;
	argTimeOutput->mins=(n % 60);
	addHours(argTimeOutput, (n/60));
	}
}

void addSeconds(TIME_T* argTimeOutput, long n)
{
	if(n>0)
	{
	n+=(long)argTimeOutput->secs;
	argTimeOutput->secs=(n % 60);
	addMinutes(argTimeOutput, (n/60));
	}
}

#if(IS_PC_HOST)
unsigned long scaleTimeSeconds(double timeSpeed, unsigned long offset, unsigned long timeToScale)
{
	double x;
	
	x=(((double)timeToScale-(double)offset)*(double)timeSpeed)+(double)offset;
	return (unsigned long)x;
}

void scaleTime(double timeSpeed, unsigned long offset, TIME_T* timeToScale)
{
	unsigned long absolute;
	absolute=getTotalSeconds(timeToScale);
	absolute=scaleTimeSeconds(timeSpeed, offset, absolute);
	convertEpoch2TimeInternal(absolute, timeToScale);
}
#endif

void checkTime(TIME_T* argTimeOutput)
{
	/* Checks that the time makes sense! */
	if(argTimeOutput->year<1970)
	{
		argTimeOutput->year=1970;
	}	
	
	if((argTimeOutput->secs>=0)&&(argTimeOutput->secs<60))
	{
	
	}
	else
	{
		argTimeOutput->secs=(argTimeOutput->secs % 60);
	}
	
	if((argTimeOutput->mins>=0)&&(argTimeOutput->mins<60))
	{
	
	}
	else
	{
		argTimeOutput->mins=(argTimeOutput->mins % 60);
	}
	
	if((argTimeOutput->hours>=0)&&(argTimeOutput->hours<24))
	{
	
	}
	else
	{
		argTimeOutput->hours=(argTimeOutput->hours % 24);
	}
	
	if(argTimeOutput->year<1970)
	{
		argTimeOutput->year=1970;
	}
	
	if((argTimeOutput->month>=1)&&(argTimeOutput->month<=12))
	{
	
	}
	else
	{
		argTimeOutput->month=(argTimeOutput->month % 12)+1;
	}
	
	if((argTimeOutput->day>=1)&&(argTimeOutput->month<=daysInMonth(argTimeOutput->month, argTimeOutput->year)))
	{
	
	}
	else
	{
		argTimeOutput->day=(argTimeOutput->day % (daysInMonth(argTimeOutput->month, argTimeOutput->year)))+1;
	}	
}
			
unsigned int convertBCDToDecimal(unsigned int bcdValue)
{
	unsigned int x;
	x=0;
	x=((bcdValue>>4)&0x0F)*10;
	x+=(bcdValue&0x0F);
	return x;
}

unsigned int convertDecimalToBCD(unsigned int decValue)
{
	unsigned int x;

	x=0x0F & (decValue/10);
	x=x<<4;
	x+=(decValue % 10);
	return (x & 0xFF);
}

unsigned int isLeapYear(unsigned int iyear)
{
	// returns TRUE if iyear is a leap year, FALSE otherwise
	int i, j;
	
	if(iyear>=1970)
	{
		i=(iyear%4);
		if(i!=0)return 0;	// since not divisible by 4
		i=(iyear%100);
		if(i!=0)return 1;
		/* here it is divisible by 4 and 100 but not necessarily by 400! */
		j=(iyear%400);
		if(j==0)return 1; else return 0;
	}
	return 0;
}

unsigned int daysInMonth(unsigned int month, unsigned int year)
{
	unsigned int i;
	
	if(month<1)month=1; else if(month>12)month=12;
	if(isLeapYear(year)&&(month==2))i=1; else i=0;
	return i+monthDayLimits[month-1];
}

void convertToHoursMins(TIME_T* inTime, float f, int base)
{
	int i;
	int h, m;
	
	i=(float)f;
	if(i<0)i=0; else if(i>=9999)i=9999;
	h=(i/base);
	m=(i%base);
	inTime->hours=h;
	inTime->mins=m;	
}

float convertFromHoursMins(TIME_T* inTime, int base)
{
	return (float)((inTime->hours*base)+(inTime->mins));
}

void convertEpoch2TimeInternal(unsigned long epochTime, TIME_T* dest)
{
    int correction;
    int i;
    int years;
    char month;
    unsigned long day;
    int hours;
    int mins;
    int secs;
    unsigned long x;
    int yday;
    int monthday;
        
    // this function only works for 1970 epoch and only for years before 2100...
    // day is the number of days elapsed

    day = (unsigned long)((unsigned long)epochTime/(unsigned long)86400);
    // x is the remainder
    x = (unsigned long)((unsigned long)epochTime - (unsigned long)86400*(unsigned long)day);
    hours = (int) (x/3600);
    x = x - (unsigned long)3600 * (unsigned long)hours;
    mins = (int) (x/60);
    secs = (int) (x - (unsigned long)60 * (unsigned long)mins);
    years = (int)(((float)day)/(float)365.0);
    // first approximation for years...
    // yday is the day of the year...
    yday=(int)((unsigned long)day - (unsigned long)(((float)365.25*(float)years)+(float)0.25))+1;
    if(years>=131)yday++; // correction for the year 2100 which is not a leap year...
    if(yday<=0)
    { 
    years--; 
    yday+=365; 
    if(isLeapYear(1970+years))yday++;
    }
    if(isLeapYear(1970+years))correction=1; else correction=0;  
    i=10;
    while(i>0)
    {
        if(yday>(monthDays[i]+correction))break;
        i--;
    }    
    month=2+i;
    if(yday<=31)month=1;
    if(i==0)correction=0;
    if(month>1)monthday=yday-monthDays[i]-correction; else monthday=yday;
    dest->wday =(int)(((long)day+3)%7);
	dest->year= years+1970;
	dest->month = month;
	dest->day = monthday;
	dest->hours = hours;
	dest->mins = mins;
	dest->secs = secs;
}

unsigned long getTotalDays(unsigned int day, unsigned int month, unsigned int year)
{
	/* 
	return the day of the week from the day, month and year information 
	We know that 1 January 1970 was a Thursday
	We need to know how many days have elapsed since 1 January 1970...		
	*/
	unsigned int N4, N100, N400, j, i;
	unsigned long totalDays;
	if(year>=1970)
	{
		N4=(unsigned int)((year-1969)/4);
		N100=(unsigned int)((year-1901)/100);
		N400=(unsigned int)((year-1601)/400);
		totalDays=(year-1970)*365+N4-N100+N400;
		month=month-1;
		i=isLeapYear(year);
		while(month>0)
		{
			if(month==2)j=monthDayLimits[1]+i; else if(month<=12)j=monthDayLimits[month-1]; else return 0;
			totalDays+=j;
			month--;
		}	
		totalDays+=(day-1);
		return totalDays;
	}
	return 0;
}

unsigned int getWeekDay(unsigned int day, unsigned int month, unsigned int year)
{
	return ((getTotalDays(day, month, year)+3)%7);
}

unsigned long getTotalSeconds(TIME_T* timet)
{
	/* 
	
		Get total seconds from the EPOCH time 
	
	*/

	unsigned long days;
	unsigned long totalSeconds;

	days=getTotalDays(timet->day, timet->month, timet->year);
	totalSeconds=(unsigned long)86400*days;
	totalSeconds+=((unsigned long)3600*(unsigned long)timet->hours);
	totalSeconds+=((unsigned long)60*(unsigned long)timet->mins);
	totalSeconds+=(unsigned long)timet->secs;
	return totalSeconds;
}

void addMonth(TIME_T* argTimeOutput)
{
	/* adds and roughly clips the output */
	if((argTimeOutput->month>=1)&&(argTimeOutput->month<12))
	{
		argTimeOutput->month++;
	}
	else
	{
		if(argTimeOutput->month==12)argTimeOutput->year++;
		argTimeOutput->month=1;
	}
}

void addDay(TIME_T* argTimeOutput)
{
	if((argTimeOutput->day>=1)&&(argTimeOutput->day<(daysInMonth(argTimeOutput->month, argTimeOutput->year))))
	{
		argTimeOutput->day++;
	}
	else
	{
		argTimeOutput->day=1;
		addMonth(argTimeOutput);	
	}
}

void addHour(TIME_T* argTimeOutput)
{
	if((argTimeOutput->hours>=0)&&(argTimeOutput->hours<23))
	{
		argTimeOutput->hours++;
	}
	else
	{
		argTimeOutput->hours=0;
		addDay(argTimeOutput);
	}
}

void addMinute(TIME_T* argTimeOutput)
{
	if((argTimeOutput->mins>=0)&&(argTimeOutput->mins<59))
	{
		argTimeOutput->mins++;
	}
	else
	{
		argTimeOutput->mins=0;
		addHour(argTimeOutput);
	}
}

void addSecond(TIME_T* argTimeOutput)
{
	if((argTimeOutput->secs>=0)&&(argTimeOutput->secs<59))
	{
		argTimeOutput->secs++;
	}
	else
	{
		argTimeOutput->secs=0;
		addHour(argTimeOutput);
	}	
}

void getFutureArgumentTime(TIME_T* argTimeOutput, TIME_T* currentTimeInput)
{
	/* Within up to a year in the future */
	
	int i;
	
	if(argTimeOutput->show & SHOW_MONTH)
	{
	
	}
	else
	{
		argTimeOutput->month=currentTimeInput->month;	
	}
	
	if(argTimeOutput->show & SHOW_DAY)
	{
	
	}
	else
	{
		argTimeOutput->day=currentTimeInput->day;
	}
	
	if(argTimeOutput->show & SHOW_HOURS)
	{
		   
	}
	else
	{
		argTimeOutput->hours=currentTimeInput->hours;
	}
	
	if(argTimeOutput->show & SHOW_MINUTES)
	{
	
	}
	else
	{
		argTimeOutput->mins=currentTimeInput->mins;
	}
	
	if(argTimeOutput->show & SHOW_SECONDS)
	{
	
	}
	else
	{
		argTimeOutput->secs=currentTimeInput->secs;
	}
	
	argTimeOutput->year=currentTimeInput->year;
	argTimeOutput->show&=~SHOW_YEAR;
	
	/* Now we need to compute whether the argTime is in the future or not, and add carry over time if not */

	/* The following returns 0 if equal, 1 if timeOne is after timeTwo, -1 if timeTwo is after timeOne */
	i=compareTimes(argTimeOutput, currentTimeInput);
	if(i>0)
	{
		/* if strictly in the future, ok */
	}
	else
	{
		/* not in the future, so we must carry over until it is... */
		if(argTimeOutput->show & SHOW_MONTH)
		{
			/* the month was specified, but still it was in the past, so we must add a year! */
			argTimeOutput->year++;
		}
		else
		if(argTimeOutput->show & SHOW_DAY)
		{
			/* the day was specified, but the month was not, and still in the past, so add a month */
			addMonth(argTimeOutput);
		}
		else
		if(argTimeOutput->show & SHOW_HOURS)
		{
			/* the hour was specified, but the day and month were not, so add a day */
			addDay(argTimeOutput);
		}
		else
		if(argTimeOutput->show & SHOW_MINUTES)
		{
			/* the minutes were specified, but the day, month and hour were not, so add an hour */
			addHour(argTimeOutput);
		}
		else
		if(argTimeOutput->show & SHOW_SECONDS)
		{
			/* the seconds were specified, but the day, month, hour and minutes were not, so add a minute */
			addMinute(argTimeOutput);		
		}
		else
		{
			/* nothing was specified, so return one more second than the currentInputTime */
			addSecond(argTimeOutput);			
		}
	}	
}

long getDurationTotalSeconds(TIME_T* timeOne, TIME_T* timeTwo)
{
	unsigned long seconds1;
	unsigned long seconds2;
	
	seconds1=getTotalSeconds(timeOne);
	seconds2=getTotalSeconds(timeTwo);
	return seconds1-seconds2;
}	
	
unsigned char* internalAscTime(unsigned char* instr, TIME_T* timet, unsigned char normal)
{
	BYTE* outstr;
	
	/* 
		convert a TIME_T structure to a string...
	
	*/
	
	outstr=instr;
	
	if(timet->show & SHOW_WDAY)
	{
	timet->wday=getWeekDay(timet->day, timet->month, timet->year);
	if(timet->wday<0)timet->wday=0;
	else
	if(timet->wday>6)timet->wday=6;
	*outstr++=weekDays[timet->wday][0];
	*outstr++=weekDays[timet->wday][1];
	*outstr++=weekDays[timet->wday][2];
	}
	
	if((normal)||(timet->show & SHOW_DAY))
	{
		*outstr++=' ';
		outstr=myultoa(timet->day, outstr, 0);
	}
	
	if((normal)||(timet->show & SHOW_MONTH))
	{
	*outstr++=' ';
	if(timet->month<1)timet->month=1;
	else
	if(timet->month>12)timet->month=12;	
	*outstr++=monthNames[timet->month-1][0];
	*outstr++=monthNames[timet->month-1][1];
	*outstr++=monthNames[timet->month-1][2];
	}
	
	if((normal)||(timet->show & SHOW_YEAR))
	{
	*outstr++=' ';
	outstr=(BYTE*)myultoa(timet->year, outstr, 0);
	}
	
	if((normal)||(timet->show & SHOW_HOURS))
	{
	*outstr++=' ';
	outstr=(BYTE*)myultoa(timet->hours, outstr, 2);
	}
	if((normal)||(timet->show & SHOW_MINUTES))
	{
	*outstr++=':';
	outstr=(BYTE*)myultoa(timet->mins, outstr, 2);
	}
	if((normal)||(timet->show & SHOW_SECONDS))
	{
	*outstr++=':';
	outstr=(BYTE*)myultoa(timet->secs, outstr, 2);
	}

	*outstr='\0';
	return outstr;
}

unsigned char* ascTime(unsigned char* instr, TIME_T* timet)
{
	return internalAscTime(instr, timet, 0);
}

unsigned char* ascTimeNormal(unsigned char* instr, TIME_T* timet)
{
	return internalAscTime(instr, timet, 1);
}

unsigned char* ascTimeFileName(unsigned char* instr, TIME_T* timet)
{
	BYTE* outstr;
	/* 
		convert a TIME_T structure to a string suitable for a filename...
	*/
	
	outstr=instr;
	if(timet->show & SHOW_YEAR)outstr=myultoa(timet->year, outstr, 4);
	if(timet->show & SHOW_MONTH)outstr=myultoa(timet->month, outstr, 2);
	if(timet->show & SHOW_DAY)outstr=myultoa(timet->day, outstr, 2);
	if(timet->show & SHOW_HOURS)outstr=myultoa(timet->hours, outstr, 2);
	if(timet->show & SHOW_MINUTES)outstr=myultoa(timet->mins, outstr, 2);
	if(timet->show & SHOW_SECONDS)outstr=myultoa(timet->secs, outstr, 2);
	*outstr='\0';
	return outstr;
}

#if 0
unsigned char* getFileNameTimeString(unsigned char* outstring, rom const unsigned char* baseString, TIME_T* timet)
{
	outstring=copyStringRom(outstring, baseString);
	outstring=ascTimeFileName(outstring, timet);
	return outstring;
}
#endif

BYTE* getTimeStringSystemTime(BYTE* outstr)
{
	return ascTime(outstr, (TIME_T*)&systemTime); 
}

#if 0
unsigned char* ascTimeLog(unsigned char* instr, TIME_T* timet)
{
	BYTE* outstr;
	
	/* 
		convert a TIME_T structure to a string...
	
	*/
	
	outstr=instr;
	if(timet->show & SHOW_WDAY)
	{
	timet->wday=getWeekDay(timet->day, timet->month, timet->year);
	if(timet->wday<0)timet->wday=0;
	else
	if(timet->wday>6)timet->wday=6;
	*outstr++=weekDays[timet->wday][0];
	*outstr++=weekDays[timet->wday][1];
	*outstr++=weekDays[timet->wday][2];
	}
	if(timet->show & SHOW_DAY)
	{
	*outstr++=',';
	outstr=myultoa(timet->day, outstr, 0);
	}
	if(timet->show & SHOW_MONTH)
	{
	*outstr++=',';
	if(timet->month<1)timet->month=1;
	else
	if(timet->month>12)timet->month=12;	
	*outstr++=monthNames[timet->month-1][0];
	*outstr++=monthNames[timet->month-1][1];
	*outstr++=monthNames[timet->month-1][2];
	}
	if(timet->show & SHOW_YEAR)
	{
	*outstr++=',';
	outstr=(BYTE*)myultoa(timet->year, outstr, 0);
	}
	if(timet->show & SHOW_HOURS)
	{
	*outstr++=',';
	outstr=(BYTE*)myultoa(timet->hours, outstr, 2);
	}
	if(timet->show & SHOW_MINUTES)
	{
	*outstr++=',';
	outstr=(BYTE*)myultoa(timet->mins, outstr, 2);
	}
	if(timet->show & SHOW_SECONDS)
	{
	*outstr++=',';
	outstr=(BYTE*)myultoa(timet->secs, outstr, 2);
	}
	*outstr='\0';
	return outstr;
}

BYTE* getTimeStringLog(BYTE* outstr)
{
	return ascTimeLog(outstr, (TIME_T*)&systemTime); 
}
#endif

unsigned long getDuration(TIME_T* timeOne, TIME_T* timeTwo)
{
	/* get seconds between two times disregarding dates */
	long total;
	int x;
	
  	total=0;
	if(timeTwo)
	{
	if(timeOne->hours>=timeTwo->hours)x=(timeOne->hours-timeTwo->hours); else x=24+(timeOne->hours-timeTwo->hours);
	total+=(long)(x*24*60);
	x=(timeOne->mins-timeTwo->mins);
	total+=(long)(x*60);
	x=(timeOne->secs-timeTwo->secs);
	total+=(long)x;
	}
	else
	{
	x=(timeOne->hours);
	total+=(long)(x*24*60);
	x=(timeOne->mins);
	total+=(long)(x*60);
	x=(timeOne->secs);
	total+=(long)x;	
	}
	if(total<0)total=-total;
	return (unsigned long)total;
}

int compareTimes(TIME_T* timeOne, TIME_T* timeTwo)
{
	/* returns 0 if equal, 1 if timeOne is after timeTwo, -1 if timeTwo is after timeOne */
	if(timeOne->show & SHOW_YEAR)
	{
		/* compare years */
		if(timeOne->year>timeTwo->year)return 1;
		else
		if(timeTwo->year>timeOne->year)return -1;	
	}
	
	if(timeOne->show & SHOW_MONTH)
	{
		/* compare months */
		if(timeOne->month>timeTwo->month)return 1;	
		else
		if(timeTwo->month>timeOne->month)return -1;	
	}
	
	if(timeOne->show & SHOW_DAY)
	{
		/* compare days */
		if(timeOne->day>timeTwo->day)return 1;	
		else
		if(timeTwo->day>timeOne->day)return -1;	
	}
	
	if(timeOne->show & SHOW_HOURS)
	{
		/* compare hours */
		if(timeOne->hours>timeTwo->hours)return 1;	
		else
		if(timeTwo->hours>timeOne->hours)return -1;	
	}
	
	if(timeOne->show & SHOW_MINUTES)
	{
		/* compare mins */
		if(timeOne->mins>timeTwo->mins)return 1;
		else
		if(timeTwo->mins>timeOne->mins)return -1;	
	}
	
	if(timeOne->show & SHOW_SECONDS)
	{
		/* compare seconds */
		if(timeOne->secs>timeTwo->secs)return 1;	
		else
		if(timeTwo->secs>timeOne->secs)return -1;	
	}
	return 0;
}

unsigned char* getDurationString(unsigned char* outstr, float floatSeconds, int printed)
{
	/* 
	 	update the time string with minutes in double...
	 
	 */
	 	
	unsigned long days;
	unsigned int hours;
	unsigned int minutes;
	unsigned int seconds;
	unsigned char sign;
	
	if(floatSeconds<0.0)
	{
		floatSeconds=-floatSeconds;
		sign=1;
	}
	else
	{
		sign=0;
	}
	
	days=(unsigned long)(floatSeconds/86400.0);
	floatSeconds-=((float)days*86400.0);
	hours=(unsigned int)(floatSeconds/3600.0);
	floatSeconds-=((float)hours*3600.0);
	minutes=(unsigned int)(floatSeconds/60.0);
	floatSeconds-=((float)minutes*60.0);
	seconds=(unsigned int)(floatSeconds);
	if((days>0)||(printed))
	{
	if(sign)
	{
	*outstr++='-';
	sign=0;
	}
	outstr=(BYTE*)myultoa((long)days, outstr, 0);
	*outstr++='d';
	*outstr++=' ';
	printed=1;
	}
	
	if((hours>0)||(printed))
	{
	if(sign)
	{
	*outstr++='-';
	sign=0;
	}
	outstr=(BYTE*)myultoa((long)hours, outstr, 0);
	*outstr++='h';
	*outstr++=' ';
	printed=1;
	}
	
	if((minutes>0)||(printed))
	{
	if(sign)
	{
	*outstr++='-';
	sign=0;
	}
	outstr=(BYTE*)myultoa((long)minutes, outstr, 0);
	*outstr++='m';
	*outstr++=' ';
	printed=1;
	}
	
	if(sign)
	{
	*outstr++='-';
	sign=0;
	}
	outstr=(BYTE*)myultoa((long)seconds, outstr, 0);
	*outstr++='s';
	*outstr='\0';
	return outstr;
}

int setAlarmTime(TIME_T* sysTime)
{
	unsigned char mask;
	unsigned char oldGIE;

 	oldGIE = INTCONbits.GIE;
    INTCONbits.GIE = 0;
    EECON2 = 0x55;
    EECON2 = 0xAA;
    RTCCFGbits.RTCWREN=1;
    INTCONbits.GIE=oldGIE;
      
	 /* disable the alarm temporarily, as it can cause false triggering otherwise */
	ALRMCFGbits.ALRMEN=0;		
	
	alarmEvent=CLEAR_ALARM_EVENT;
	
	ALRMCFGbits.ALRMPTR0=1;
	ALRMCFGbits.ALRMPTR1=1;
	
	ALRMVALL=(unsigned char)convertDecimalToBCD(sysTime->year-EPOCH_YEAR);
	ALRMVALH=0;
	
	ALRMVALL=(unsigned char)convertDecimalToBCD(sysTime->day);
	ALRMVALH=(unsigned char)convertDecimalToBCD(sysTime->month);
	
	ALRMVALL=(unsigned char)convertDecimalToBCD(sysTime->hours);
	ALRMVALH=(unsigned char)convertDecimalToBCD(sysTime->wday);
	
	ALRMVALL=(unsigned char)convertDecimalToBCD(sysTime->secs);
	ALRMVALH=(unsigned char)convertDecimalToBCD(sysTime->mins);
	
	ALRMRPT=0x00;		/* no repeat of the alarm */
	
	if(sysTime->show & SHOW_YEAR)
	{
		mask=0x24;
	}
	else
	if(sysTime->show & SHOW_MONTH)
	{
		mask=0x24;	
	}
	else
	if(sysTime->show & SHOW_DAY)
	{
	 	mask=0x20;	
	}
	else
	if(sysTime->show & SHOW_HOURS)
	{
		mask=0x18;
	}
	else
	if(sysTime->show & SHOW_MINUTES)
	{
	  	mask=0x14;	
	}
	else
	{
	  	mask=0x0C;
	}
	
	PIR3bits.RTCCIF=0;
	IPR3bits.RTCCIP=0;
	PIE3bits.RTCCIE=1;
	
	ALRMCFG=(0x80 + mask);		/* enable the alarm, no chime, every year mask */

 	oldGIE = INTCONbits.GIE;
    INTCONbits.GIE = 0;
    EECON2 = 0x55;
    EECON2 = 0xAA;
    RTCCFGbits.RTCWREN=0;
    INTCONbits.GIE=oldGIE;
	
	return 0;
}

int setTime(TIME_T* sysTime)
{
	unsigned char oldALR;
	unsigned char oldGIE;
	
	oldALR = ALRMCFGbits.ALRMEN;
	oldGIE = INTCONbits.GIE;
    INTCONbits.GIE = 0;
    EECON2 = 0x55;
    EECON2 = 0xAA;
    RTCCFGbits.RTCWREN=1;
    INTCONbits.GIE=oldGIE;

	/* a write to RTCEN is only allowed when RTCWREN (bit 5 of RTCCFG) is 1 */
	RTCCFGbits.RTCEN=0;
	ALRMCFGbits.ALRMEN=0;
	
	/* disable the alarm temporarily, as it can cause false triggering otherwise */
	ALRMCFGbits.ALRMEN=0;		
	
	RTCCFGbits.RTCPTR0=1;
	RTCCFGbits.RTCPTR1=1;
	
	RTCVALL=(unsigned char)convertDecimalToBCD(sysTime->year-EPOCH_YEAR);
	RTCVALH=0;
	
	RTCVALL=(unsigned char)convertDecimalToBCD(sysTime->day);
	RTCVALH=(unsigned char)convertDecimalToBCD(sysTime->month);
	
	RTCVALL=(unsigned char)convertDecimalToBCD(sysTime->hours);
	RTCVALH=(unsigned char)convertDecimalToBCD(sysTime->wday);
	
	RTCVALL=(unsigned char)convertDecimalToBCD(sysTime->secs);
	RTCVALH=(unsigned char)convertDecimalToBCD(sysTime->mins);
	
	ALRMCFGbits.ALRMEN=oldALR;
	RTCCFGbits.RTCEN=1;
	
	oldGIE = INTCONbits.GIE;
    INTCONbits.GIE = 0;
    EECON2 = 0x55;
    EECON2 = 0xAA;
    RTCCFGbits.RTCWREN=0;
    INTCONbits.GIE=oldGIE;
	timeUp=TIME_UPDATED;
	return 0;
}

int igetTime(TIME_T* sysTime)
{
	unsigned int value;
	
	RTCCFGbits.RTCPTR0=1;
	RTCCFGbits.RTCPTR1=1;
	
	value=(unsigned int)RTCVALL;
	
	value+=((unsigned int)RTCVALH & 0);
	
	sysTime->year=(unsigned int)EPOCH_YEAR+convertBCDToDecimal(value);
	
	value=(unsigned int)RTCVALL;
	sysTime->day=convertBCDToDecimal(value);
	
	value=(unsigned int)RTCVALH;
	sysTime->month=convertBCDToDecimal(value);
	
	value=(unsigned int)RTCVALL;
	sysTime->hours=convertBCDToDecimal(value);
	
	value=(unsigned int)RTCVALH;
	sysTime->wday=convertBCDToDecimal(value);
	
	value=(unsigned int)RTCVALL;
	sysTime->secs=convertBCDToDecimal(value);
	
	value=(unsigned int)RTCVALH;
	sysTime->mins=convertBCDToDecimal(value);
	sysTime->updated|=TIME_UPDATED;
	return 0;
}

int getTime(ram TIME_T* sysTime)
{
	TIME_T iTime;
	
	/* used for the comparison ... */
	
	if((RTCCFGbits.RTCSYNC==0)&&(RTCCFGbits.HALFSEC==0))
	{
		igetTime(sysTime);
	}
	else
	{
		igetTime(sysTime);
		igetTime(&iTime);
		if(!((iTime.secs==sysTime->secs)
		&&
		(iTime.mins==sysTime->mins)
		&&
		(iTime.hours==sysTime->hours)
		&&
		(iTime.day==sysTime->day)
		&&
		(iTime.month==sysTime->month)
		&&
		(iTime.year==sysTime->year)))
		{
			igetTime(sysTime);
		}
	}
	return 0;
}

#if 0
void doTimer1Interrupt(void)
{
	if(PIR1bits.TMR1IF)
	{
		if(hostAlive>0)hostAlive--;
		timer1OverFlow++;
		PIR1bits.TMR1IF=0;
	}
}
#endif

int initRTCC(TIME_T* sysTime)
{	
	unsigned char oldGIE;
	/* Initialize the real time clock module... */

	if(timeUp)
	{
		
	}
	else
	{
			
	#if(USE_PMDIS)
		PMDIS1bits.TMR1MD=0;
		PMDIS1bits.RTCCMD=0;
	#endif
	
	alarmEvent=CLEAR_ALARM_EVENT;
	timer1OverFlow=0;
	/* 
	   Note: Bit 2 of T1CON (the bar-TSYNC bit) MUST be low for the capture objects that use TMR1 to work properly, as
	   it forces the timer 1 to be synchronised with the processors clock, this makes sense because crystals X1 and X2
	   can be not synchronised...) (Mauro Grassi)
	*/
	T1CON=0x8B;
	IPR1bits.TMR1IP=1;
	PIR1bits.TMR1IF=0;
	PIE1bits.TMR1IE=1;
	IPR3bits.RTCCIP=0;
	
	PADCFG1bits.RTSECSEL1=0;
	PADCFG1bits.RTSECSEL0=1;
	
	/* disable output pin */
	RTCCFGbits.RTCOE=0;
	//ISBbits.TRISB1=1;
	
	oldGIE = INTCONbits.GIE;
    INTCONbits.GIE = 0;
    EECON2 = 0x55;
    EECON2 = 0xAA;
    RTCCFGbits.RTCWREN=1;
    INTCONbits.GIE=oldGIE;
    
	/* a write to RTCEN is only allowed when RTCWREN (bit 5 of RTCCFG) is 1 */
	RTCCFGbits.RTCEN=1;
	ALRMCFGbits.ALRMEN=0;
	
	oldGIE = INTCONbits.GIE;
    INTCONbits.GIE = 0;
    EECON2 = 0x55;
    EECON2 = 0xAA;
    RTCCFGbits.RTCWREN=0;
    INTCONbits.GIE=oldGIE;
    
  	sysTime->updated&=~TIME_UPDATED;
	sysTime->updated&=~DATE_UPDATED;
	sysTime->secs=0;
	sysTime->mins=0;
	sysTime->hours=0;
	sysTime->day=1;
	sysTime->month=1;
	sysTime->year=2011;
	sysTime->wday=5;
	sysTime->show=(SHOW_TIME_DEFAULT);
	setTime(sysTime);
	/* The following is needed because setTime sets timeUp to 1! */
	timeUp=0;
	}
	return 0;
}

void copyTime(TIME_T* timeDest, TIME_T* timeSource)
{
	timeDest->hours=timeSource->hours;
	timeDest->mins=timeSource->mins;
	timeDest->secs=timeSource->secs;
	timeDest->day=timeSource->day;
	timeDest->wday=timeSource->wday;
	timeDest->month=timeSource->month;
	timeDest->year=timeSource->year;
	timeDest->show=timeSource->show;
	timeDest->updated=timeSource->updated;
}

#if 0
unsigned long getSecondsFromDynamicObject(DYNAMIC_DURATION_OBJECT* dynObj)
{
	/* convert from Dynamic Duration Object to seconds */
	unsigned long result;

	if(dynObj->dynamicDuration>=142)
	{
		result=(unsigned long)((unsigned long)(dynObj->dynamicDuration-141)*86400);
	}
	else
	if(dynObj->dynamicDuration>=119)
	{
		result=(unsigned long)((unsigned long)(dynObj->dynamicDuration-118)*3600);
	}
	else
	if(dynObj->dynamicDuration>=60)
	{
		result=(unsigned long)((unsigned long)(dynObj->dynamicDuration-59)*60);
	}
	else
	{
		result=(unsigned long)(dynObj->dynamicDuration);
	}
	return result;
}

void getDynamicObjectFromSeconds(DYNAMIC_DURATION_OBJECT* dynObj, unsigned long seconds)
{
	/* 
		convert from seconds (from 0 to 114 days) 
		
		to Dynamic Duration Object 
	
		Note that 114 days = 86400 * 114 = 9849600
			
	*/

	unsigned int  x;
	unsigned char result;
	
	if(seconds<MAX_DYNAMIC_DURATION_OBJECT_SECONDS)
	{
		x=(unsigned int)(seconds/86400);
		/* x is number of days */
		if(x>0)
		{
			result=x+141;		
		}
		else
		{
			x=(unsigned int)(seconds/3600);
			/* x is number of hours */
			if(x>0)
			{
				result=x+118;			
			}
			else
			{
				x=(unsigned int)(seconds/60);
				/* x is number of minutes */
				if(x>0)
				{
					result=x+59;
				}
				else
				{
					result=x;
				}
			}
		}
	}
	else
	{
		/* clip any duration above 114 days */
		result=0xFF;
	}
	dynObj->dynamicDuration=result;
}

#endif
